/* 
 * This file is part of the Tucana Echo2 Library.
 * Copyright (C) 2007.
 *
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 */
 
/**
 * Static namespace for the FileUploadSelector.
 */
TucanaFileUploadSelector = function(elementId, containerId) {
    this.selectorId = elementId;
    this.containerId = containerId;
};

/***********************
 * Firefox Width hack  *
 ***********************/
TucanaFileUploadSelector.needsWidthHack = function() {
    if (EchoClientProperties.get("browserMozillaFirefox") ||
        EchoClientProperties.get("browserMozilla")) {
        return true;
    }
    return false;
};

TucanaFileUploadSelector.calculateWidthHackFunction = function() {
    if (!TucanaFileUploadSelector.widthHackSlope) {
       var testDiv = document.createElement("div");
       testDiv.style.position = "absolute";
       testDiv.style.left = "-10000px";
       testDiv.style.opacity = "0";
       var testForm = document.createElement("form");
       testDiv.appendChild(testForm);
       var testInput = document.createElement("input");
       testForm.appendChild(testInput);
       
       document.body.appendChild(testDiv);

       var smallSize = 5;
       var largeSize = 300;

       testInput.setAttribute("size", smallSize.toString());
       var smallWidth = testInput.offsetWidth;
       testInput.setAttribute("size", largeSize.toString());
       var largeWidth = testInput.offsetWidth;
       
       var slope = (largeWidth - smallWidth) / (largeSize - smallSize);

       var intercept = smallWidth - (slope / smallSize);
       
       TucanaFileUploadSelector.widthHackSlope = slope;
       TucanaFileUploadSelector.widthHackIntercept = intercept;
       
//       EchoDebugManager.consoleWrite("Slope: " + slope);
//       EchoDebugManager.consoleWrite("Intercept: " + intercept);
       

       document.body.removeChild(testDiv);
    }
};

TucanaFileUploadSelector.getSizeForWidth = function(width) {
    // width = m(size) + b
    // size = (width-b)/m
    var size = (width - TucanaFileUploadSelector.widthHackIntercept) / 
       TucanaFileUploadSelector.widthHackSlope;
    size = parseInt(size);
    size = (size < 1) ? 1 : size;
//    EchoDebugManager.consoleWrite("Calculated size: " + size);
    return size;
};

TucanaFileUploadSelector.prototype.updateWidthHackSize = function() {
    var targetWidth = this.elements.divInput.offsetWidth;
    if (!this.widthHack.oldTargetWidth || this.widthHack.oldTargetWidth != targetWidth) {
        var newSize = TucanaFileUploadSelector.getSizeForWidth(targetWidth);
        this.elements.input.setAttribute("size", newSize.toString());
        while (newSize > 1 && this.elements.divInput.scrollWidth > this.elements.divInput.offsetWidth) {
            newSize--;
            this.elements.input.setAttribute("size", newSize.toString());
        }
//        EchoDebugManager.consoleWrite("Size set to: " + newSize);
        this.widthHack.oldTargetWidth = targetWidth;
    }
};

TucanaFileUploadSelector.prototype.enableWidthHack = function() { 
    if (TucanaFileUploadSelector.needsWidthHack()) {
        var instance = this;
        this.elements.divInput.style.overflow = "hidden";
        if (this.widthHack.interval != null) {
            clearInterval(this.widthhack.interval);
        }
	    this.widthHack.interval = setInterval(function() { instance.updateWidthHackSize(); }, 10);
    }
};

TucanaFileUploadSelector.prototype.disableWidthHack = function() { 
    this.elements.divInput.style.overflow = "";
    if (this.widthHack.interval) {
        clearInterval(this.widthHack.interval);
        this.widthHack.interval = null;
    }
    this.widthHack.oldTargetWidth = null;
};


/***********************
 * Utilities           *
 ***********************/
TucanaFileUploadSelector.getInstance = function(elementId) {
    return EchoDomPropertyStore.getPropertyValue(elementId, "instance");
};

TucanaFileUploadSelector.prototype.clearInput = function() {
    if (EchoClientProperties.get("browserMozillaFirefox") ||
        EchoClientProperties.get("browserMozilla") ||
        EchoClientProperties.get("browserOpera")) {
        this.elements.form.reset();
		this.elements.input.disabled = false;
    } else {
        var newInput = this.elements.input.cloneNode(true);
        this.elements.divInput.replaceChild(newInput, this.elements.input);
        this.elements.input = newInput;
    }
};

TucanaFileUploadSelector.getAutoDisplayType = function() {
    if (EchoClientProperties.get("browserSafari")) {
        return "left";
    } else {
        return "right";
    }
};

/**
 * Set the opacity of an element.
 * 
 * @param element The element whose opacity to change.
 * @param value The opacity value (0.0 - 1.0)
 */
TucanaFileUploadSelector.setOpacity = function(element, value) {
	if (EchoClientProperties.get("browserInternetExplorer")) {
		element.style.zoom = 1;
		element.style.filter = "alpha(opacity=" + value*100 + ")";
	} else {
		element.style.opacity = value;
	}
};

/***********************
 * Progress components *
 ***********************/
TucanaFileUploadSelector.prototype.getProgressPollerType = function() {
	if (TucanaFileUploadSelector.ListenerManager.getListeners(this.selectorId) == null) {
		return "none";
	} else if (EchoClientProperties.get("browserOpera")) {
        return "sync";
    } else {
        return "async";
    }
};

TucanaFileUploadSelector.prototype.createProgressPoller = function(progressChangeHandler) {
    var instance = this;
    this.uploadSession.progress = 0;
    if (this.getProgressPollerType() == "sync") {
        return function() {
            var progressElement = parent.frames[instance.uploadSession.frameId].document.body.lastChild;
            if (progressElement && progressElement.id != "finished") {
                if (instance.uploadSession) {
					var bytesElement = progressElement.childNodes[0];
					var totalBytesElement = progressElement.childNodes[1];
					var rateElement = progressElement.childNodes[2];
					
					var bytes = null;
					var totalBytes = null;
					var rate = null;
					
					if (bytesElement) {
						bytes = parseInt(bytesElement.firstChild.nodeValue)
					}
					if (totalBytesElement) {
						totalBytes = parseInt(totalBytesElement.firstChild.nodeValue)
					}
					if (rateElement) {
						rate = parseFloat(rateElement.firstChild.nodeValue)
					}

					progressChangeHandler(bytes, totalBytes, rate);
                }
            }
        };
    } else if (this.getProgressPollerType() == "async") {
        var connectionLock = function() {};
        connectionLock.connecting = false;

        var handler = function(connection) {
            connectionLock.connecting = false;
            var resp = connection.getResponseXml();
            var ts = resp.firstChild;
            if (instance.uploadSession) {
				var bytesElement = ts.childNodes[0];
				var totalBytesElement = ts.childNodes[1];
				var rateElement = ts.childNodes[2];
				var bytes = null;
				var totalBytes = null;
				var rate = null;
				
				if (bytesElement) {
					bytes = parseInt(bytesElement.firstChild.nodeValue)
				}
				if (totalBytesElement) {
					totalBytes = parseInt(totalBytesElement.firstChild.nodeValue)
				}
				if (rateElement) {
					rate = parseFloat(rateElement.firstChild.nodeValue)
				}

				progressChangeHandler(bytes, totalBytes, rate);
            }
        };

        return function() {
            if (connectionLock.connecting == false) {
                connectionLock.connecting = true;
                var connection = new EchoHttpConnection("?serviceId=Tucana.FileUploadReceiver&sid=" + instance.selectorId, "GET", null, null);
                connection.responseHandler = handler;
                connection.connect();
            }
        };
    } else {
    	return null;
    }
};

/***********************
 * File submission     *
 ***********************/
 
 /*
  * First step: Initialize the submission frame.
  */
TucanaFileUploadSelector.prototype.initializeSubmit = function() {
	
	if (this.uploadSession) {
		if (this.uploadSession.supportsCancel()) {
			EchoDebugManager.consoleWrite("CANCELING");
			// upload in progress, cancel.
			this.uploadSession.cancel();
		}
		return false;
	}
	
    EchoDebugManager.consoleWrite("HANDLESUBMIT");
	this.uploadSession = function() {};

	var instance = this;
	var us = this.uploadSession;
	// Upload session funcs
	this.uploadSession.cleanUp = function() {
		var successStatus = null;
		try {
			successStatus = parent.frames[us.frameId].document.body.lastChild.firstChild.nodeValue;
		} catch (e) {
		}
		if (successStatus == null) {
			successStatus = "fail";
		}

		var frameContainer = us.frameContainer;
		if (us.poller != null) {
			clearInterval(us.poller);
			us.poller = null;
		}
		
		if (us.progressHandler != null && us.progressHandler.finish != null) {
		    us.progressHandler.finish();
		    us.progressHandler = null;
		}

		instance.clearInput();
		
	    /* If we remove the IFrame in this thread, Firefox displays a halfway-busy
	     * cursor indefinitely. Removing the frame in a 0ms timeout fixes this issue.
	     */
	    if (EchoClientProperties.get("browserMozillaFirefox")) {
	        setTimeout(function() { 
	            frameContainer.parentNode.removeChild(frameContainer);
	        }, 0);
	    } else {
	    	if (frameContainer) {
	        	frameContainer.parentNode.removeChild(frameContainer);
	    	}
	    }
	    
	    if (EchoClientProperties.get("browserKonqueror")) {
	    	// reset konq url
	    	parent.frames[us.frameId].window.location.href = "?serviceId=Tucana.FileUploadSelector.IFrame";
	    }
		EchoDebugManager.consoleWrite("Upload complete: " + successStatus);

		instance.updateButton(false);
	};
	
	this.uploadSession.cancel = function() {
		instance.uploadSession = null;
		if (us.frame != null) {
            var frameWindow = parent.frames[us.frameId].window;
//				EchoDebugManager.consoleWrite(frameWindow);
			if (frameWindow.stop) {
				EchoDebugManager.consoleWrite("WINDOW DOT STOP");
		    	frameWindow.stop();
			} else if (frameWindow.document && frameWindow.document.execCommand) {
				EchoDebugManager.consoleWrite("EXECCOMMAND STOP");
				frameWindow.document.execCommand("Stop");
			}
			else if (EchoClientProperties.get("browserKonqueror")) {
				frameWindow.location.href = "?serviceId=Tucana.FileUploadSelector.IFrame";
			}
		}
		us.cleanUp();
	};
	
	this.uploadSession.supportsCancel = function() {
		if (!instance.button.cancelEnabled) {
			return false;
		}
		if (us.frame != null) {
            var frameWindow = parent.frames[us.frameId].window;
				EchoDebugManager.consoleWrite(frameWindow);
			// window.stop()
			if (frameWindow.stop) {
				return true;
			} 
			// document.execCommand("Stop")
			else if (frameWindow.document && frameWindow.document.execCommand) {
				return true;
			} 
			// window.location.href = ...
			else if (EchoClientProperties.get("browserKonqueror")) {
				return true;
			}
		}
		return false;
	}

    var divElm = document.createElement('div');
    divElm.style.position = "absolute";
    divElm.style.top = "0";
    divElm.style.marginLeft = "-10000px";
    var frameId = "tucanaUploadSelectorFrame_" + this.selectorId + "_" 
    	+ (this.uploadCount++).toString();

    /* IE Needs this ugly hack, of course. */
    if (EchoClientProperties.get("browserInternetExplorer")) {
        divElm.innerHTML = 
        '<iframe src=\"?serviceId=Tucana.FileUploadSelector.IFrame\" name=\"'
        + frameId 
        + '\" id=\"' 
        + frameId
        +'\" onload=\"TucanaFileUploadSelector.continueSubmit(\''
        + this.selectorId 
        + '\');\"></iframe>';
    	this.uploadSession.frame = divElm.firstChild;
    	document.getElementsByTagName("body").item(0).appendChild(divElm);
    	this.uploadSession.frameContainer = divElm;
    } else if (EchoClientProperties.get("browserKonqueror")) {
    /* When using konqueror, we keep an IFrame around with the selector */
    	frameId = this.elements.iframe.getAttribute("id");
    	this.uploadSession.frame = this.elements.iframe;
    } else {
        var frame = document.createElement("iframe");
        frame.setAttribute("src", "?serviceId=Tucana.FileUploadSelector.IFrame");
        frame.setAttribute("name", frameId);
        frame.setAttribute("id", frameId);
        /* Safari doesn't appear to give the initial onload */
//        if (!EchoClientProperties.get("browserSafari")) {
            var instance = this;
            EchoDomUtil.addEventListener(frame, "load", function() { 
            	EchoDomUtil.removeEventListener(frame, "load", arguments.callee, false); 
            	instance.continueSubmit(); 
            }, false);
//        }
        divElm.appendChild(frame);
    	this.uploadSession.frame = frame;
        document.getElementsByTagName("body").item(0).appendChild(divElm);
        this.uploadSession.frameContainer = divElm;
    }
    this.uploadSession.frameId = frameId;
   	
   	/* Safari doesn't fire the first onload event, and Konqueror has its iframe
   	 * created way before (so we miss the onload event).  Therefore, both of 
   	 * these are fired here.
   	 */
    if (/*EchoClientProperties.get("browserSafari") ||*/
    	EchoClientProperties.get("browserKonqueror")) {
        this.continueSubmit();
    }
};

/*
 * Second step: submit to the frame
 */
TucanaFileUploadSelector.prototype.continueSubmit = function() {
	if (this.uploadSession.alreadyContinued) {
		return;
	}
	this.uploadSession.alreadyContinued = true;

    EchoDebugManager.consoleWrite("CONTINUESUBMIT");
    var instance = this;
	var frame = this.uploadSession.frame;
    EchoDomUtil.addEventListener(frame, "load", function() {
    	instance.handleSubmitComplete();
    }, false);


    var pollTime = -1;
    if (this.getProgressPollerType() == "sync") {
        pollTime = this.syncReportInterval;
    } else if (this.getProgressPollerType() == "async") {
        pollTime = this.asyncPollInterval;
    }
    
    var us = this.uploadSession;
    
    if (pollTime != -1) {
        var selectorId = this.selectorId;
	    us.totalBytes = 1;
	    us.rate = 0;
	    var progressHandler = function(bytes, totalBytes, rate) {
            us.totalBytes = totalBytes;
            us.rate = rate;
            var listeners = 
                TucanaFileUploadSelector.ListenerManager.getListeners(selectorId);
            for (var i in listeners) {
                var func = listeners[i];
                if (func != null) {
                    func(bytes, totalBytes, rate);
                }
            }
	    };
	    progressHandler.finish = function() {
            var b = (us.totalBytes != 0) ? us.totalBytes : 1;
            progressHandler(b, b, us.rate);	        
	    };
	    progressHandler(0, 0, 0);
    	var poller = this.createProgressPoller(progressHandler);
    	this.uploadSession.poller = setInterval(poller, pollTime);
    	this.uploadSession.progressHandler = progressHandler;
    }

    this.elements.form.target = this.uploadSession.frameId;
    this.elements.form.setAttribute("target", this.uploadSession.frameId);
    this.elements.form.submit();
	this.updateButton(true);
//    this.elements.input.disabled = true;
};

/*
 * Third step: clean up after submission finishes
 */
TucanaFileUploadSelector.prototype.handleSubmitComplete = function() {

	if (!this.uploadSession) {
		// canceled
		return;
	}
	/*
	 * IE reaches this method as soon as the form is submited, and then again
	 * when the submission completes.  We only want to do stuff the second time.
	 */
	if (EchoClientProperties.get("browserInternetExplorer")) {
		if (!this.uploadSession.firstCompleteReached) {
			this.uploadSession.firstCompleteReached = true;
			return;
		}
	}
	
    EchoDebugManager.consoleWrite("HANDLESUBMITCOMPLETE");

	this.uploadSession.cleanUp();
	this.uploadSession = null;
	
	/* Sync with server */
	EchoClientMessage.setActionValue(this.selectorId, "fileUploaded");
    EchoServerTransaction.connect();

};

/*
 * Static accessors for submission methods
 */
TucanaFileUploadSelector.initializeSubmit = function(selectorId) {
	var instance = TucanaFileUploadSelector.getInstance(selectorId);
    instance.initializeSubmit();
	return false;
};

TucanaFileUploadSelector.continueSubmit = function(selectorId) {
	var instance = TucanaFileUploadSelector.getInstance(selectorId);
	instance.continueSubmit();
};

// Lifecycle

TucanaFileUploadSelector.prototype.create = function() {
    if (EchoClientProperties.get("browserMozillaFirefox")) {
        TucanaFileUploadSelector.calculateWidthHackFunction();
    }
	this.uploadCount = 0;
	this.syncReportInterval = 200;
	this.asyncPollInterval = 500;

    this.elements = function() {};
    this.button = function() {};
    this.width = function() {};
    this.widthHack = function() {};
    this.progress = function() {};

    this.elements.selector = document.createElement("div");
    
    this.elements.selector.setAttribute("id", this.selectorId);
    this.elements.container = document.getElementById(this.containerId);
    
    /* IE makes us do it this way */
    if (EchoClientProperties.get("browserInternetExplorer")) {
        this.elements.form = document.createElement("<form enctype='multipart/form-data'/>");
    } else {
        this.elements.form = document.createElement("form");
        this.elements.form.setAttribute("enctype", "multipart/form-data");
    }

    this.elements.form.style.position = "relative";
    this.elements.form.style.margin = "0px";
    this.elements.form.setAttribute("method", "post");
    this.elements.form.setAttribute("action", "?serviceId=Tucana.FileUploadReceiver&sid=" + 
        this.selectorId + "&pt=" + this.getProgressPollerType() +
        "&ri=" + this.syncReportInterval);
    this.elements.selector.appendChild(this.elements.form);

    this.elements.table = document.createElement("table");
    this.elements.table.style.borderCollapse = "collapse";
    this.elements.table.appendChild(document.createElement("thead"));
    this.elements.tbody = document.createElement("tbody");
    this.elements.table.appendChild(this.elements.tbody);
    this.elements.tr = document.createElement("tr");
    this.elements.tbody.appendChild(this.elements.tr);
    this.elements.form.appendChild(this.elements.table);

    this.elements.tdSubmitLeft = document.createElement("td");
    this.elements.tdSubmitLeft.style.display = "none";
    this.elements.tdSubmitLeft.style.padding = "0px 2px 0px 0px";
    this.elements.tr.appendChild(this.elements.tdSubmitLeft);

    this.elements.tdInput = document.createElement("td");
    this.elements.tdInput.style.padding = "0px";
    this.elements.tr.appendChild(this.elements.tdInput);
    this.elements.divInput = document.createElement("div");
    this.elements.tdInput.appendChild(this.elements.divInput);
    this.elements.input = document.createElement("input");
    this.elements.input.setAttribute("type", "file");
    this.elements.input.setAttribute("name", "$$file$$");
    this.elements.divInput.appendChild(this.elements.input);
    
    this.elements.tdSubmitRight = document.createElement("td");
    this.elements.tdSubmitRight.style.display = "none";
    this.elements.tdSubmitRight.style.padding = "0px 0px 0px 2px";
    this.elements.tr.appendChild(this.elements.tdSubmitRight);

    this.elements.submit = document.createElement("input");

	/* Konqueror doesn't work when creating the iframes on the spot.. */
	if (EchoClientProperties.get("browserKonqueror")) {    
		var frameId = this.selectorId + "_iframe";
		this.elements.iframe = document.createElement("iframe");
        this.elements.iframe.src = "?serviceId=Tucana.FileUploadSelector.IFrame";
		this.elements.iframe.style.position = "absolute";
		this.elements.iframe.style.left = "-10000px";
		this.elements.iframe.setAttribute("id", frameId);
		this.elements.selector.appendChild(this.elements.iframe);
    }
    
    this.elements.container.appendChild(this.elements.selector);
    
    EchoDomPropertyStore.setPropertyValue(this.elements.selector, "instance", this);

    var instance = this;
	EchoDomUtil.addEventListener(this.elements.form, "submit", function(e) { 
		instance.initializeSubmit(); 
		if (e && e.preventDefault) {
			e.preventDefault();
		}
		return false;
	}, false);

};

TucanaFileUploadSelector.prototype.updateButton = function(uploading) {
   var parent = this.elements.submit.parentNode;
   if (parent) {
       parent.removeChild(this.elements.submit);
       this.elements.submit = document.createElement("input");
   }
   if (this.button.mode == "image") {
       var imageSource;
	   if (uploading) {
	       if (this.uploadSession.supportsCancel()) {
	           imageSource = this.button.cancelImageSource;
	       } else {
	       	   imageSource = this.button.waitImageSource;
	       }
	   } else {
	       imageSource = this.button.uploadImageSource;
	   }
       this.elements.submit.style.height = null;
       this.elements.submit.setAttribute("type", "image");
       this.elements.submit.setAttribute("src", imageSource);
   } else if (this.button.mode == "text") {
       var text;
	   var disabled = false;
	   if (uploading) {
	       if (this.uploadSession.supportsCancel()) {
	           text = this.button.cancelText;
	       } else {
	       	   text = this.button.waitText;
	       	   disabled = true;
	       }
	   } else {
	       text = this.button.uploadText;
	   }
       this.elements.submit.setAttribute("type", "submit");
       this.elements.submit.style.height = this.elements.input.offsetHeight + "px";
       if (text == null) {
           this.elements.submit.removeAttribute("value");
       } else {
           this.elements.submit.setAttribute("value", text)
       }
       this.elements.submit.disabled = disabled;
   }
   if (parent) {
       parent.appendChild(this.elements.submit);
   }
}

TucanaFileUploadSelector.prototype.processProperties = function(propertiesElement) {
    var updateButton = false;
    var updateWidth = false;
    var updateButtonDisplay = false;

    for (var i = 0; i < propertiesElement.childNodes.length; i++) {
        var property = propertiesElement.childNodes[i];
        var propertyName = property.getAttribute("name");
        var propertyValue = property.getAttribute("value");
//        EchoDebugManager.consoleWrite("PROPERTY: " + propertyName + "=" + propertyValue);
        switch (propertyName) {
        	case "button-mode":
                updateButton = true;
        	    if (propertyValue != null) {
            	    var mode = parseInt(propertyValue);
        	        if (mode == 1) {
        	            this.button.mode = "image";
        	        } else {
        	            this.button.mode = "text";
        	        }
        	    } else { /* Default value */
        	        this.button.mode = "text";
        	    }
        		break;

        	case "button-text-upload":
                updateButton = true;
                this.button.uploadText = propertyValue;
        		break;

        	case "button-text-cancel":
                updateButton = true;
                this.button.cancelText = propertyValue;
        		break;

        	case "button-text-wait":
                updateButton = true;
                this.button.waitText = propertyValue;
        		break;
        	
            case "button-image-upload":
                updateButton = true;
                this.button.uploadImageSource = propertyValue;
                break;

            case "button-image-cancel":
                updateButton = true;
                this.button.cancelImageSource = propertyValue;
                break;

            case "button-image-wait":
                updateButton = true;
                this.button.waitImageSource = propertyValue;
                break;
                
            case "button-display":
                updateButtonDisplay = true;
        	    if (propertyValue != null) {
            	    var mode = parseInt(propertyValue);
                    if (mode == 0) {
                        this.button.display = "right";
                    } else if (mode == 1) {
                        this.button.display = "left";
                    } else if (mode == 2) {
                        this.button.display = "auto";
                    } else if (mode == 3) {
                        this.button.display = "none";
                    } else {
                        this.button.display = "auto";
                    }
                } else { /* Default value */
                    this.button.display = "auto";
                }     
                break;
                
            case "width-mode": 
                updateWidth = true;
                if (propertyValue != null) {
                    var widthMode = parseInt(propertyValue);
                    if (widthMode == 1) {
                        this.width.mode = "extent";
                    } else {
                        this.width.mode = "size";
                    }
                } else { /* Default value */
                    this.width.mode = "size";
                }
                break;
                
            case "width-extent":
                updateWidth = true;
                this.width.extent = propertyValue;
                break;
                
            case "width-size":
                updateWidth = true;
                this.width.size = propertyValue;
                break;

            case "cancel-enabled":
            	this.button.cancelEnabled = ("true" == propertyValue);
				updateButton = true;
            	break;
        }
    }
    
    if (updateWidth) {
        if (this.width.mode == "extent") {
            this.elements.input.removeAttribute("size");
            this.elements.table.style.width = this.width.extent;
            this.elements.tdInput.style.width = "100%";
            this.elements.input.style.width = "100%";
            // If we switch to extent mode without clearing the input
            // in IE, IE makes the minimum size of the input the
            // size of the length of the text.
            if (EchoClientProperties.get("browserInternetExplorer")) {
                this.clearInput();
            }
            this.enableWidthHack();
        } else if (this.width.mode == "size") {
            this.disableWidthHack();
            this.elements.input.setAttribute("size", this.width.size);
            this.elements.table.style.width = "";
            this.elements.tdInput.style.width = "";
            this.elements.input.style.width = "";
            
            if (EchoClientProperties.get("browserOpera")) {
                /* Opera has a bug where a table's width is 100%
                 * unless we do this.
                 */
                var parent = this.elements.table.parentNode;
                parent.removeChild(this.elements.table);
                parent.appendChild(this.elements.table);
            }
        }
    }
    
    if (updateButton) {
		this.updateButton(this.uploadSession != null);
    }
    
    if (updateButtonDisplay) {
        var displayType;
        if (this.button.display == "auto") {
            displayType = TucanaFileUploadSelector.getAutoDisplayType();                
        } else {
            displayType = this.button.display;
        }
        
        if (displayType == "left") {
            this.elements.tdSubmitLeft.appendChild(this.elements.submit);
            this.elements.tdSubmitLeft.style.display = "";
            this.elements.tdSubmitRight.style.display = "none";
            /* Opera has some bug that puts space between the submit
             * button and the input box unless we do this.
             */
            if (EchoClientProperties.get("browserOpera")) {
                this.elements.tdSubmitLeft.width = "1px";
            }
        } else if (displayType == "right") {
            this.elements.tdSubmitRight.appendChild(this.elements.submit);
            this.elements.tdSubmitRight.style.display = "";
            this.elements.tdSubmitLeft.style.display = "none";
        } else if (displayType == "none") {
        }
    }

};

TucanaFileUploadSelector.prototype.dispose = function() {
	this.disableWidthHack();
	this.elements.container.removeChild(this.elements.selector);
	if (this.elements.iframe && this.elements.iframe.parentNode) {
//        EchoDebugManager.consoleWrite("Removing Konq IFrame");
        this.elements.iframe.parentNode.removeChild(this.elements.iframe);
        this.elements.iframe = null;
	}
	if (this.uploadSession) {
		this.uploadSession.cancel();
	}
    this.elements = null;	    
};


// Message Processor

/**
 * Namespace fr erver->client message processor
 */
TucanaFileUploadSelector.MessageProcessor = function() { };

/** Static message processing method.  This method delegates to specific handlers.
 * 
 * @param messagePartElement message element to process
 */
TucanaFileUploadSelector.MessageProcessor.process = function(messagePartElement) {
    for (var i = 0; i < messagePartElement.childNodes.length; ++i) {
        if (messagePartElement.childNodes[i].nodeType == 1) {
            switch (messagePartElement.childNodes[i].tagName) {
            case "init":
                TucanaFileUploadSelector.MessageProcessor.processInit(messagePartElement.childNodes[i]);
                break;
            case "dispose":
                TucanaFileUploadSelector.MessageProcessor.processDispose(messagePartElement.childNodes[i]);
                break;
            case "update":
                TucanaFileUploadSelector.MessageProcessor.processUpdate(messagePartElement.childNodes[i]);
                break;
            }
        }
    }
};

/**
 * Processes a <code>dispose</code> message to finalize the state of a
 * selector component that is being removed.
 *
 * @param disposeMessageElement the <code>dispose</code> element to process
 */
TucanaFileUploadSelector.MessageProcessor.processDispose = function(disposeMessageElement) {
	var elementId = disposeMessageElement.getAttribute("eid");
	var instance = TucanaFileUploadSelector.getInstance(elementId);
	if (instance != null) {
		instance.dispose();
	} else {
		EchoDebugManager.consoleWrite("Dispose message for invalid id: " + elementId);
	}
};

/**
 * Processes an <code>init</code> message to initialize the state of a 
 * selector component that is being added.
 *
 * @param initMessageElement the <code>init</code> element to process
 */
TucanaFileUploadSelector.MessageProcessor.processInit = function(initMessageElement) {
    var elementId = initMessageElement.getAttribute("eid");
    var containerId = initMessageElement.getAttribute("container-eid");
    var propertiesElement = initMessageElement.getElementsByTagName("properties")[0];
    var selector = new TucanaFileUploadSelector(elementId, containerId);
	selector.create();
	selector.processProperties(propertiesElement);
};

TucanaFileUploadSelector.MessageProcessor.processUpdate = function(updateMessageElement) {
    var elementId = updateMessageElement.getAttribute("eid");
    var propertiesElement = updateMessageElement.getElementsByTagName("properties")[0];
    var instance = TucanaFileUploadSelector.getInstance(elementId);
    if (instance != null) {
        instance.processProperties(propertiesElement);
    } else {
        EchoDebugManager.consoleWrite("Update message for invalid id: " + elementId);
    }
};

/**
 * File upload listener manager
 * Manages listening components, such as progress bars
 */
TucanaFileUploadSelector.ListenerManager = function() {};
TucanaFileUploadSelector.ListenerManager.listenerMap = new Object();

TucanaFileUploadSelector.ListenerManager.addListener = function(uploaderId, listenerId, listenerFunc) {
    var uploaderListeners = TucanaFileUploadSelector.ListenerManager.listenerMap[uploaderId];
    if (uploaderListeners == null) {
        uploaderListeners = new Object();
    	TucanaFileUploadSelector.ListenerManager.listenerMap[uploaderId] = uploaderListeners;
    }
    
    uploaderListeners[listenerId] = listenerFunc;
};

TucanaFileUploadSelector.ListenerManager.removeListener = function(uploaderId, listenerId) {
    var uploaderListeners = TucanaFileUploadSelector.ListenerManager.listenerMap[uploaderId];
	if (uploaderListeners != null) {
		delete uploaderListeners[listenerId];
		if (uploaderListeners.length == 0) {
			delete TucanaFileUploadSelector.ListenerManager.listenerMap[uploaderId];
		}
	}
};

TucanaFileUploadSelector.ListenerManager.getListeners = function(uploaderId) {
	return TucanaFileUploadSelector.ListenerManager.listenerMap[uploaderId];
};